//Weather Station
//The DIY Life by Michael Klements
//11 August 2021

#include <Wire.h>                                                   //Import the required libraries
#include "DHT.h"
#include "Seeed_BMP280.h"
#include <WiFi.h>
#include "ThingSpeak.h"

#define lightSen 36                                                 //Define pin numbers
#define windSen 0
#define uS_TO_S_FACTOR 1000000ULL                                   //Conversion factor for micro seconds to seconds
#define TIME_TO_SLEEP  570                                          //Time ESP32 will go to sleep (in seconds) - 9.5 minutes

DHT dht(14,DHT11);                                                  //DHT and bMP sensor parameters
BMP280 bmp280;

WiFiClient client;                                                  //WiFi connection details
char ssid[] = "WIFI SSID";                                          //WiFi Name
char pass[] = "WIFI PASSWORD";                                      //WiFi Password

unsigned long myChannelNumber = YOUR CHANNEL ID;                    //Thingspeak channel number
const char * myWriteAPIKey = "YOUR WRITE API KEY";                  //Thingspeak API write key

int light = 0;                                                      //Variables to store sensor readings
int temp = 0;
int humid = 0;
int pressure = 0;
int wind = 0;

unsigned long firstMillis = 0;                                       //Timers for the wind speed calculation
unsigned long lastMillis = 0;
unsigned long lastIntTime = 0;
int counter = 0;                                                     //Counter to keep track of the number of wind speed revolutions 

void IRAM_ATTR isr ()                                                //Interrupt routine, run with each reed switch interrupt
{
  unsigned long intTime = millis();
  if(intTime - lastIntTime > 150)                                    //Debounce the reed switch input
  {
    if (counter == 0)
      firstMillis = millis();
    counter++;                                                       //Count each revolution
    lastMillis = millis();
    Serial.println(counter);
  }
  lastIntTime = intTime;                                             //Capture the first and last revolution time
}

void setup()                                                         //Setup function - only function that is run in deep sleep mode
{
  Serial.begin(9600);
  if(!bmp280.init())                                                 //Connect to pressure sensor
  {
    Serial.println("bmp280 init error!");
  }
  pinMode(lightSen, INPUT);                                          //Define pin functions
  pinMode(windSen, INPUT_PULLUP);
  attachInterrupt(windSen, isr, FALLING);                            //Define interrupt pin
  WiFi.begin(ssid, pass);                                            //Connect to WiFi network
  while (WiFi.status() != WL_CONNECTED)
  {
    delay(500);
    Serial.print(".");
  }
  Serial.println("");
  Serial.println("WiFi connected");
  esp_sleep_enable_timer_wakeup(TIME_TO_SLEEP * uS_TO_S_FACTOR);     //Initialise deep sleep mode parameters
  ThingSpeak.begin(client);                                          //Initialise ThingSpeak
  delay(10000);                                                      //Wait for wind speed readings to be taken
  recLight ();                                                       //Take readings from other sensors
  recTempHumid ();
  recPress ();
  calcWind ();
  counter = 0;
  Serial.print("Light: ");                                           //Display readings on serial monitor
  Serial.println(light);
  Serial.print("Temp: ");
  Serial.println(temp);
  Serial.print("Humidity: ");
  Serial.println(humid);
  Serial.print("Pressure: ");
  Serial.println(pressure);
  Serial.print("Wind: ");
  Serial.println(wind);
  updateThingSpeak ();                                               //Post the data to Thingspeak
  Serial.println("Going to sleep now");
  Serial.flush(); 
  esp_deep_sleep_start();                                            //Enter sleep mode
}

void loop()                                                          //Loop function - unused
{

}

void recLight ()                                                     //Function to record the light level
{
  light = analogRead(lightSen);
}

void recTempHumid ()                                                 //Function to record the temperature and humidity
{
  temp = dht.readTemperature();
  humid = dht.readHumidity();
}

void recPress ()                                                     //Function to record the pressure
{
  pressure = bmp280.getPressure()/100;
}

void calcWind ()                                                     //Function to calculate the wind speed
{
  int ave = 5000;
  if(counter != 0)
    ave = (lastMillis - firstMillis)/counter;
  Serial.print("Average Tick Time: ");
  Serial.println(ave);
  if (ave < 200)
  {
    ave = 200;
    wind = map (ave,200, 4000, 16, 3);
  }
  else if (ave > 4000)
    wind = 0;
  else
  {
    wind = map (ave,200, 4000, 16, 3);
  }
}

void updateThingSpeak ()                                             //Function to post data to Thingspeak
{
  ThingSpeak.setField(1, light);
  ThingSpeak.setField(2, temp);
  ThingSpeak.setField(3, humid);
  ThingSpeak.setField(4, pressure);
  ThingSpeak.setField(5, wind);
  int x = ThingSpeak.writeFields(myChannelNumber, myWriteAPIKey);
  if(x == 200)
  {
    Serial.println("Channel update successful.");
  }
  else
  {
    Serial.println("Problem updating channel. HTTP error code " + String(x));
  }
}
