//Weather Station V2
//The DIY Life by Michael Klements
//25 September 2021

#include <Wire.h>                                                   //Import the required libraries
#include <Adafruit_Sensor.h>
#include <Adafruit_BME280.h>
#include <WiFi.h>
#include "ThingSpeak.h"

#define sensPow1 16                                                 //Define pin numbers
#define sensPow2 17
#define lightSen 36
#define windSen 0
#define uS_TO_S_FACTOR 1000000ULL                                   //Conversion factor for micro seconds to seconds
#define TIME_TO_SLEEP  20                                           //Time ESP32 will go to sleep (in seconds) - 9.5 minutes / 590 seconds

Adafruit_BME280 bme;                                                // I2C

WiFiClient client;                                                  //WiFi connection details
char ssid[] = "????";                                               //WiFi Name
char pass[] = "????";                                               //WiFi Password

unsigned long myChannelNumber = ????;                               //Thingspeak channel number
const char * myWriteAPIKey = "????";                                //Thingspeak API write key

int light = 0;                                                      //Variables to store sensor readings
int temp = 0;
int humid = 0;
int pressure = 0;
int wind = 0;

int windMap[2][32] = {{2000,1400,1000,600,500,400,320,280,240,220,200,180,170,160,150,140,130,120,110,100,90,80,70,60,50,40,38,36,34,32,30,28},{0,1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16,17,18,20,23,25,29,34,41,51,54,57,60,64,68,73}};

unsigned long startTime = 0;
unsigned long endTime = 0;
int counter = 0;                                                     //Counter to keep track of the number of wind speed revolutions 

void setup()                                                         //Setup function - only function that is run in deep sleep mode
{
  Serial.begin(9600);
  pinMode(sensPow1, OUTPUT);                                         //Define power pins function
  pinMode(sensPow2, OUTPUT);
  digitalWrite(sensPow1, HIGH);                                      //Switch power to the BME sensor on
  digitalWrite(sensPow2, HIGH);                                      //Switch power to the light & hall effect sensor on
  pinMode(lightSen, INPUT);                                          //Define pin functions
  pinMode(windSen, INPUT_PULLUP);
  counter = 0;                                                       //Reset wind speed counter
  esp_sleep_enable_timer_wakeup(TIME_TO_SLEEP * uS_TO_S_FACTOR);     //Initialise deep sleep mode parameters
  unsigned status;
  delay(200);                                                        //Allow BME sensor to startup
  status = bme.begin();                                              //Connect to BME sensor
  if (!status)
  {
    Serial.println("Could not find a valid BME280 sensor");
  }
  int lastState = 0;                                                 //Variable to store the last state of wind sensor input
  for (int i=0; i<= 6000; i++)
  {
    if (digitalRead(windSen) == HIGH && lastState == 0)
    {
      counter++;
      lastState = 1;
    }
    else if (digitalRead(windSen) == LOW && lastState == 1)
    {
      lastState = 0;
    }
    delay(1);
  }
  recTempHumid ();                                                   //Take readings from BME sensor
  recPress ();
  digitalWrite(sensPow1, LOW);                                       //Switch power to the BME sensor off
  delay(200);                                                        //Wait for sensor LED to turn off
  recLight ();                                                       //Take readings from the light sensor (done once BME sensor is off to prevent LED interferance)
  calcWind ();
  digitalWrite(sensPow2, LOW);                                       //Switch power to the light & hall effect sensor off
  Serial.print("Light: ");                                           //Display readings on serial monitor
  Serial.println(light);
  Serial.print("Temp: ");
  Serial.println(temp);
  Serial.print("Humidity: ");
  Serial.println(humid);
  Serial.print("Pressure: ");
  Serial.println(pressure);
  Serial.print("Wind: ");
  Serial.println(wind);
  WiFi.begin(ssid, pass);                                            //Connect to WiFi network
  int timeout = 0;
  while (WiFi.status() != WL_CONNECTED  && timeout < 20)
  {
    delay(500);
    Serial.print(".");
    timeout++;
  }
  if(WiFi.status() == WL_CONNECTED)
  {
    Serial.println("");
    Serial.println("WiFi connected");
    ThingSpeak.begin(client);                                        //Initialise ThingSpeak
    updateThingSpeak ();                                             //Post the data to Thingspeak
  }
  else
  {
    Serial.println("");
    Serial.println("WiFi connection failed");
  }
  Serial.println("Going to sleep now");
  Serial.flush();
  esp_deep_sleep_start();                                            //Enter sleep mode
}

void loop()                                                          //Loop function - unused
{

}

void recLight ()                                                     //Function to record the light level
{
  light = analogRead(lightSen);
}

void recTempHumid ()                                                 //Function to record the temperature and humidity
{
  temp = bme.readTemperature();
  humid = bme.readHumidity();
}

void recPress ()                                                     //Function to record the pressure
{
  pressure = bme.readPressure()/100;
}

void calcWind ()                                                     //Function to calculate the wind speed
{
  int ave = 2000;
  if(counter > 1)                                                    //Check that at least two ticks have been registered, one is meaningless
  {
    ave = 6000/(counter-1);                                          //Calculate the average time between ticks
  }
  Serial.print("Average Tick Time: ");
  Serial.println(ave);
  if (ave < 28)                                                      //Discard readings faster than 102kph
    ave = 28;
  int index = 0;
  while (ave < windMap[0][index])                                    //Search through the windMap array for the corressponding wind value
    index++;
  wind = windMap[1][index];
  Serial.print("Wind Speed: ");
  Serial.println(wind);
}

void updateThingSpeak ()                                             //Function to post data to Thingspeak
{
  ThingSpeak.setField(1, light);
  ThingSpeak.setField(2, temp);
  ThingSpeak.setField(3, humid);
  ThingSpeak.setField(4, pressure);
  ThingSpeak.setField(5, wind);
  int x = ThingSpeak.writeFields(myChannelNumber, myWriteAPIKey);
  if(x == 200)
  {
    Serial.println("Channel update successful.");
  }
  else
  {
    Serial.println("Problem updating channel. HTTP error code " + String(x));
  }
}
