//MakerFabs Weather Station with 4G
//The DIY Life by Michael Klements
//4 December 2021

#include <stdio.h>                                                      //Include the required libraries
#include <string.h>
#include <DHT.h>
#include <Seeed_BMP280.h>

String Apikey = "?????";                                                //Thingspeak write API key

#define DEBUG true                                                      //Define LTE parameters and pin functions
#define LTE_RESET_PIN 6
#define LTE_PWRKEY_PIN 5
#define LTE_FLIGHT_PIN 7

#define pinInterrupt A0                                                 //Wind speed sensor interrupt pin
#define PM_READ_PIN A1                                                  //Particle sensor reading pin
#define PM_LED_PIN 10                                                   //Particle sensor LED pin

DHT dht(13,DHT11);                                                      //DHT and BMP sensor parameters
BMP280 bmp280;

int windCount = 0;                                                      //Counter for each anemometer revolution
int speed_value = 0;
int pm25_value = 0;
unsigned long lastDebounceTime = 0;                                     //The last time the output pin was toggled
unsigned long debounceDelay = 1000;                                     //The debounce time; increase if the output flickers

void setup()
{
    SerialUSB.begin(115200);                                            //Start serial communication for debugging
    Serial1.begin(115200);
  
    pinMode(LTE_RESET_PIN, OUTPUT);                                     //Set up LTE pin modes and initial states
    digitalWrite(LTE_RESET_PIN, LOW);
    pinMode(LTE_PWRKEY_PIN, OUTPUT);
    digitalWrite(LTE_RESET_PIN, LOW);
    delay(100);
    digitalWrite(LTE_PWRKEY_PIN, HIGH);
    delay(2000);
    digitalWrite(LTE_PWRKEY_PIN, LOW);
    pinMode(LTE_FLIGHT_PIN, OUTPUT);
    digitalWrite(LTE_FLIGHT_PIN, LOW);                                  //Normal Mode

    pinMode(pinInterrupt, INPUT_PULLUP);                                //Define pin mode and interrupt routine for wind speed sensor
    attachInterrupt(digitalPinToInterrupt(pinInterrupt), onChange, FALLING);

    pinMode(PM_LED_PIN, OUTPUT);                                        //Define pin mode for PM sensor's LED
    
    delay(5000);                                                        //Wait 5 seconds

    dht.begin();                                                        //Start DHT sensor
    if(!bmp280.init())                                                  //Connect to pressure sensor
      Serial.println("bmp280 init error!");
    
    delay(1000);                                                        //Wait 1 second for sensors to start
}

void loop()
{
    float humidity = dht.readHumidity();                               //Record the humidity
    float temperature = dht.readTemperature();                         //Record the temperature
    float pressure = bmp280.getPressure()/100;                         //Record pressure
    SerialUSB.print("Temp: ");
    SerialUSB.print(temperature);
    SerialUSB.println("C");
    SerialUSB.print("Humidity: ");
    SerialUSB.print(humidity);
    SerialUSB.println("%");
    SerialUSB.print("Pressure: ");
    SerialUSB.print(pressure);
    SerialUSB.println("hPa");
    measureWind();                                                     //Calculate the wind speed
    pm25();                                                            //Measure the PM2.5 value

    String http_str = "AT+HTTPPARA=\"URL\",\"https://api.thingspeak.com/update?api_key=" + Apikey + "&field1=" + (String)temperature + "&field2=" + (String)humidity + "&field3=" + (String)pressure + "&field4=" + (String)speed_value + "&field5=" + (String)pm25_value + "\"\r\n";
    //SerialUSB.println(http_str);                                     //Print to serial monitor for debugging

    sendData("AT+HTTPINIT\r\n", 2000, DEBUG);                          //Send the data to Thingspeak
    sendData(http_str, 2000, DEBUG);
    sendData("AT+HTTPACTION=0\r\n", 3000, DEBUG);
    sendData("AT+HTTPTERM\r\n", 3000, DEBUG);

    delay(120000);                                                      //Wait 120 seconds
}

void onChange()                                                         //Interrupt routine to increment anemometer revolution counter
{
  if (digitalRead(pinInterrupt) == LOW)
    windCount++;
}

void measureWind ()                                                     //Function to calculate the wind speed based on the revolution counts
{
  if ((millis() - lastDebounceTime) > debounceDelay)
  {
    lastDebounceTime = millis();
    speed_value = windCount * 8.75 * 0.01 / 109.56;                     //Calculate the wind speed in m/s
    SerialUSB.print(speed_value);
    windCount = 0;
    SerialUSB.println("m/s");
  }
}

void pm25()                                                             //Function to calculate the dust density
{
  int samplingTime = 280;
  int deltaTime = 40;
  int sleepTime = 9680;

  float voMeasured = 0;
  float calcVoltage = 0;
  float dustDensity = 0;
  digitalWrite(PM_LED_PIN, HIGH);                                       //Power on the LED
  delayMicroseconds(samplingTime);                                      //Wait the sampling time

  voMeasured = analogRead(PM_READ_PIN);                                 //Read the dust value
  delayMicroseconds(deltaTime);
  digitalWrite(PM_LED_PIN, LOW);                                        //Turn the LED off
  delayMicroseconds(sleepTime);

  calcVoltage = voMeasured * (3.3 / 1024.0);                            //0-5V mapped to 0-1023 integer values
  SerialUSB.print("PM2.5 Volts: ");                                     //Send value to serial monitor for debugging
  SerialUSB.print(calcVoltage);
  SerialUSB.println("V");

  dustDensity = 44.73 * calcVoltage;                                    //Calculate dust density from voltage

  SerialUSB.print("PM2.5:");                                            //Send value to serial monitor for debugging
  SerialUSB.print(dustDensity);
  SerialUSB.println("ug/m3");
  pm25_value = (int)dustDensity;
}

String sendData(String command, const int timeout, boolean debug)       //Function to send data over 4G
{
  String response = "";
  Serial1.println(command);
  
  long int time = millis();
  while ( (time + timeout) > millis())
  {
    while (Serial1.available())
    {
      char c = Serial1.read();
      response += c;
    }
  }
  if (debug)
  {
    SerialUSB.print(response);
  }
  return response;
}
